<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\HomeContent;
use Illuminate\Support\Facades\File;

class AdminHomeController extends Controller
{
    public function index()
    {
        $homeContents = HomeContent::latest()->paginate(10);
        $contentExists = HomeContent::exists();
        return view('admin.home.index', compact('homeContents', 'contentExists'));
    }

    public function create()
    {
        return view('admin.home.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'hero_heading' => 'required|string|max:255',
            'button_text' => 'required|string|max:255',
            'button_link' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:4048',
            'is_active' => 'boolean'
        ]);

        $data = $request->all();
        $data['is_active'] = $request->has('is_active');

        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            
            // Save directly to public/home
            $image->move(public_path('home'), $imageName);

            $data['image'] = $imageName;
        }

        HomeContent::create($data);

        return redirect()->route('admin.home.index')->with('success', 'Home content created successfully.');
    }

    public function show(HomeContent $home)
    {
        return view('admin.home.show', compact('home'));
    }

    public function edit(HomeContent $home)
    {
        return view('admin.home.edit', compact('home'));
    }

    public function update(Request $request, HomeContent $home)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'hero_heading' => 'required|string|max:255',
            'button_text' => 'required|string|max:255',
            'button_link' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:4048',
            'is_active' => 'boolean'
        ]);

        $data = $request->all();
        $data['is_active'] = $request->has('is_active');

        if ($request->hasFile('image')) {
            // Delete old image from public/home
            if ($home->image && File::exists(public_path('home/' . $home->image))) {
                File::delete(public_path('home/' . $home->image));
            }

            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            
            // Save new image directly to public/home
            $image->move(public_path('home'), $imageName);

            $data['image'] = $imageName;
        }

        $home->update($data);

        return redirect()->route('admin.home.index')->with('success', 'Home content updated successfully.');
    }

    public function destroy(HomeContent $home)
    {
        // Delete image from public/home
        if ($home->image && File::exists(public_path('home/' . $home->image))) {
            File::delete(public_path('home/' . $home->image));
        }

        $home->delete();

        return redirect()->route('admin.home.index')->with('success', 'Home content deleted successfully.');
    }
}
