<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Specialty;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str; //


class AdminSpecialtyController extends Controller
{
    public function index()
    {
        $specialties = Specialty::latest()->paginate(10);
        return view('admin.specialties.index', compact('specialties'));
    }

    public function create()
    {
        return view('admin.specialties.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'short' => 'required|string',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_active' => 'boolean'
        ]);

        $data = $request->all();
        $data['is_active'] = $request->has('is_active');
            $data['slug'] = Str::slug($request->short); // 🔑 generate slug


        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $image->storeAs('public/specialties', $imageName);
            $data['image'] = $imageName;
        }

        Specialty::create($data);

        return redirect()->route('admin.specialties.index')->with('success', 'Specialty created successfully.');
    }

    public function show(Specialty $specialty)
    {
        return view('admin.specialties.show', compact('specialty'));
    }

    public function edit(Specialty $specialty)
    {
        return view('admin.specialties.edit', compact('specialty'));
    }

    public function update(Request $request, Specialty $specialty)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'short' => 'required|string',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_active' => 'boolean'
        ]);

        $data = $request->all();
        $data['is_active'] = $request->has('is_active');
                    $data['slug'] = Str::slug($request->short); // 🔑 generate slug


        if ($request->hasFile('image')) {
            // Delete old image
            if ($specialty->image) {
                Storage::delete('public/specialties/' . $specialty->image);
            }

            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $image->storeAs('public/specialties', $imageName);
            $data['image'] = $imageName;
        }

        $specialty->update($data);

        return redirect()->route('admin.specialties.index')->with('success', 'Specialty updated successfully.');
    }

    public function destroy(Specialty $specialty)
    {
        // Delete image file
        if ($specialty->image) {
            Storage::delete('public/specialties/' . $specialty->image);
        }

        $specialty->delete();

        return redirect()->route('admin.specialties.index')->with('success', 'Specialty deleted successfully.');
    }
}

