<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\AdminUser;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class AdminUserController extends Controller
{
    public function index()
    {
        $adminUsers = AdminUser::latest()->paginate(10);
        return view('admin.admin-users.index', compact('adminUsers'));
    }

    public function create()
    {
        return view('admin.admin-users.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:admin_users',
            'password' => 'required|string|min:8|confirmed',
        ]);

        AdminUser::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        return redirect()->route('admin.admin-users.index')->with('success', 'Admin user created successfully.');
    }

    public function show(AdminUser $adminUser)
    {
        return view('admin.admin-users.show', compact('adminUser'));
    }

    public function edit(AdminUser $adminUser)
    {
        return view('admin.admin-users.edit', compact('adminUser'));
    }

    public function update(Request $request, AdminUser $adminUser)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('admin_users')->ignore($adminUser->id)],
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        $data = [
            'name' => $request->name,
            'email' => $request->email,
        ];

        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $adminUser->update($data);

        return redirect()->route('admin.admin-users.index')->with('success', 'Admin user updated successfully.');
    }

    public function destroy(AdminUser $adminUser)
    {
        // Prevent deleting the last admin user
        if (AdminUser::count() <= 1) {
            return redirect()->route('admin.admin-users.index')->with('error', 'Cannot delete the last admin user.');
        }

        // Prevent self-deletion
        if ($adminUser->id === auth('admin')->id()) {
            return redirect()->route('admin.admin-users.index')->with('error', 'You cannot delete your own account.');
        }

        $adminUser->delete();

        return redirect()->route('admin.admin-users.index')->with('success', 'Admin user deleted successfully.');
    }
}

