<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\BlogCategory;
use App\Models\BlogPost;
use App\Models\BlogComment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class BlogController extends Controller
{
    public function index()
    {
        $posts = BlogPost::latest()->paginate(4);
        return view('admin.blogs.index', compact('posts'));
    }

    public function create()
    {
        $categories = BlogCategory::all();
        return view('admin.blogs.create', compact('categories'));
    }

    public function store(Request $request)
    {
        // dd($request->all()); // TEMPORARY COMMENT OUT KARO

        $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|unique:blog_posts,slug',
            'short_description' => 'required|string',
            'content' => 'required|string',
            'tags' => 'required|string',
            'category_id' => 'required|exists:blog_categories,id',
            'image' => 'required|image|mimes:jpeg,png,jpg,webp|max:4048',
            // SEO Fields
            'meta_title' => 'nullable|string|max:160',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string|max:255',
            'og_title' => 'nullable|string|max:160',
            'og_description' => 'nullable|string|max:160',
            'og_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:4048',
            'canonical_url' => 'nullable|url',
            'robots' => 'nullable|string|max:150'
        ]);

        try {
            DB::beginTransaction();

            $post = new BlogPost;
            $post->title = $request->title;
            $post->slug = $request->slug;
            $post->tags = $request->tags;
            $post->short_description = $request->short_description;
            $post->content = $request->content;
            $post->category_id = $request->category_id;
            $post->author = auth('admin')->user()->name;

            // SEO Fields
            $post->meta_title = $request->meta_title ?? $request->title;
            $post->meta_description = $request->meta_description ?? $request->short_description;
            $post->meta_keywords = $request->meta_keywords ?? '';
            $post->og_title = $request->og_title ?? $request->title;
            $post->og_description = $request->og_description ?? $request->short_description;
            $post->canonical_url = $request->canonical_url;
            $post->robots = $request->robots ?? 'index, follow';

            // Main Image Handle
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $imageName = time().'_'.$image->getClientOriginalName();
                $image->move(public_path('blogs'), $imageName);
                $post->image = $imageName;
            }

            // Handle OG Image
            if ($request->hasFile('og_image')) {
                $ogImage = $request->file('og_image');
                $ogImageName = 'og_'.time().'_'.$ogImage->getClientOriginalName();
                
                // Ensure directory exists
                if (!file_exists(public_path('blogs/og'))) {
                    mkdir(public_path('blogs/og'), 0777, true);
                }
                
                $ogImage->move(public_path('blogs/og'), $ogImageName);
                $post->og_image = $ogImageName;
            }

            $post->save();
            DB::commit();

            return redirect()->route('admin.posts.index')->with('success', 'Blog post created successfully!');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->with('error', 'Error creating post: '.$e->getMessage())->withInput();
        }
    }

    public function show($slug)
    {
        $post = BlogPost::where('slug', $slug)->firstOrFail();
        $comments = BlogComment::where('post_id', $post->id)->whereNull('parent_id')->get();
    $post->increment('views');

        // Prepare SEO data for frontend
        $seoData = [
            'title' => $post->meta_title ?? $post->title,
            'description' => $post->meta_description ?? $post->short_description,
            'keywords' => $post->meta_keywords,
            'og_title' => $post->og_title ?? $post->meta_title ?? $post->title,
            'og_description' => $post->og_description ?? $post->meta_description ?? $post->short_description,
            'og_image' => $post->og_image ? asset('blogs/og/' . $post->og_image) : ($post->image ? asset('blogs/' . $post->image) : null),
            'canonical_url' => $post->canonical_url ?? url()->current(),
            'robots' => $post->robots
        ];

        return view('frontend.blogs.show', compact('post', 'comments', 'seoData'));
    }

    public function edit(BlogPost $post)
    {
        $categories = BlogCategory::all();
        return view('admin.blogs.edit', compact('post', 'categories'));
    }

    public function update(Request $request, BlogPost $post)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|unique:blog_posts,slug,' . $post->id,
            'short_description' => 'required|string',
            'content' => 'required|string',
            'tags' => 'required|string',
            'category_id' => 'required|exists:blog_categories,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:4048',
            // SEO Fields
            'meta_title' => 'nullable|string|max:60',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string|max:255',
            'og_title' => 'nullable|string|max:60',
            'og_description' => 'nullable|string|max:160',
            'og_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:4048',
            'canonical_url' => 'nullable|url',
            'robots' => 'nullable|string|max:50'
        ]);

        try {
            DB::beginTransaction();

            $post->title = $request->title;
            $post->slug = $request->slug;
            $post->tags = $request->tags;
            $post->short_description = $request->short_description;
            $post->content = $request->content;
            $post->category_id = $request->category_id;

            // SEO Fields
            $post->meta_title = $request->meta_title ?? $request->title;
            $post->meta_description = $request->meta_description ?? $request->short_description;
            $post->meta_keywords = $request->meta_keywords ?? '';
            $post->og_title = $request->og_title ?? $request->title;
            $post->og_description = $request->og_description ?? $request->short_description;
            $post->canonical_url = $request->canonical_url;
            $post->robots = $request->robots ?? 'index, follow';

            if ($request->hasFile('image')) {
                // Delete old image
                if ($post->image && file_exists(public_path('blogs/' . $post->image))) {
                    unlink(public_path('blogs/' . $post->image));
                }
                
                $image = $request->file('image');
                $imageName = time().'_'.$image->getClientOriginalName();
                $image->move(public_path('blogs'), $imageName);
                $post->image = $imageName;
            }

            // Handle OG Image update
            if ($request->hasFile('og_image')) {
                // Delete old OG image
                if ($post->og_image && file_exists(public_path('blogs/og/' . $post->og_image))) {
                    unlink(public_path('blogs/og/' . $post->og_image));
                }
                
                $ogImage = $request->file('og_image');
                $ogImageName = 'og_'.time().'_'.$ogImage->getClientOriginalName();
                
                // Ensure directory exists
                if (!file_exists(public_path('blogs/og'))) {
                    mkdir(public_path('blogs/og'), 0777, true);
                }
                
                $ogImage->move(public_path('blogs/og'), $ogImageName);
                $post->og_image = $ogImageName;
            }

            $post->save();
            DB::commit();

            return redirect()->route('admin.posts.index')->with('success', 'Blog post updated successfully!');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->with('error', 'Error updating post: '.$e->getMessage())->withInput();
        }
    }

    public function destroy(BlogPost $post)
    {
        try {
            // Delete main image
            if ($post->image && file_exists(public_path('blogs/' . $post->image))) {
                unlink(public_path('blogs/' . $post->image));
            }
            
            // Delete OG image
            if ($post->og_image && file_exists(public_path('blogs/og/' . $post->og_image))) {
                unlink(public_path('blogs/og/' . $post->og_image));
            }
            
            $post->delete();

            return back()->with('success', 'Blog post deleted successfully!');
            
        } catch (\Exception $e) {
            return back()->with('error', 'Error deleting post: '.$e->getMessage());
        }
    }
}