<?php
namespace App\Http\Controllers;

use App\Models\CourseCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class CourseCategoryController extends Controller
{
    public function index()
    {
        $categories = CourseCategory::latest()->paginate(10);
        return view('admin.course_category.index', compact('categories'));
    }

    public function create()
    {
        return view('admin.course_category.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:course_categories,name',
            'description' => 'nullable|string',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'meta_keywords' => 'nullable|string',
            'icon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:4048',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer',
        ]);

        $data = $request->all();

        // ✅ Save image directly to public folder
        if ($request->hasFile('icon')) {
            $icon = $request->file('icon');
            $iconName = time() . '_' . $icon->getClientOriginalName();
            $icon->move(public_path('categories/icons'), $iconName);
            $data['icon'] = 'categories/icons/' . $iconName;
        }

        CourseCategory::create($data);

        return redirect()->route('admin.course_category.index')
            ->with('success', 'Category created successfully.');
    }

    public function show(CourseCategory $category)
    {
        return view('admin.course_category.show', compact('category'));
    }

    public function edit(CourseCategory $category)
    {
        return view('admin.course_category.edit', compact('category'));
    }

    public function update(Request $request, CourseCategory $category)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:course_categories,name,' . $category->id,
            'description' => 'nullable|string',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'meta_keywords' => 'nullable|string',
            'icon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:4048',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer',
        ]);

        $data = $request->all();

        // ✅ Handle icon update
        if ($request->hasFile('icon')) {
            if ($category->icon && File::exists(public_path($category->icon))) {
                File::delete(public_path($category->icon));
            }

            $icon = $request->file('icon');
            $iconName = time() . '_' . $icon->getClientOriginalName();
            $icon->move(public_path('categories/icons'), $iconName);
            $data['icon'] = 'categories/icons/' . $iconName;
        }

        $category->update($data);

        return redirect()->route('admin.course_category.index')
            ->with('success', 'Category updated successfully.');
    }

    public function destroy(CourseCategory $category)
    {
        if ($category->courses()->count() > 0) {
            return redirect()->route('admin.course_category.index')
                ->with('error', 'Cannot delete category with associated courses.');
        }

        if ($category->icon && File::exists(public_path($category->icon))) {
            File::delete(public_path($category->icon));
        }

        $category->delete();

        return redirect()->route('admin.course_category.index')
            ->with('success', 'Category deleted successfully.');
    }
}
