<?php

namespace App\Http\Controllers;

use App\Models\Course;
use App\Models\CourseCategory;
use App\Models\Teacher;
use Illuminate\Http\Request;

class CourseServicesController extends Controller
{
    // CourseController.php mein yeh method add karein
public function frontendIndex(Request $request)
{
    $query = Course::with(['category', 'teacher'])
        ->where('is_active', true);

    // 🔍 Search functionality
    if ($request->filled('search')) {
        $search = $request->search;
        $query->where(function ($q) use ($search) {
            $q->where('title', 'like', "%{$search}%")
              ->orWhere('description', 'like', "%{$search}%")
              ->orWhereHas('teacher', function ($q) use ($search) {
                  $q->where('name', 'like', "%{$search}%");
              });
        });
    }

    // 🏷️ Category filter
    if ($request->filled('category')) {
        $query->whereHas('category', function ($q) use ($request) {
            $q->where('id', $request->category);
        });
    }

    // 📚 Level filter
    if ($request->filled('level')) {
        $query->where('level', $request->level);
    }

    // 💰 Price type filter
    if ($request->filled('price_type')) {
        $query->where('price_type', $request->price_type);
    }

    // ⏱ Duration filter
// Duration filter
// Duration filter
if ($request->filled('duration')) {
    switch ($request->duration) {
        case 'hours':
            $query->where('duration_type', 'hours');
            break;
        case 'days':
            $query->where('duration_type', 'days');
            break;
        case 'weeks':
            $query->where('duration_type', 'weeks');
            break;
        case 'months':
            $query->where('duration_type', 'months');
            break;
    }
}


    // 🔽 Sort functionality
    if ($request->filled('sort')) {
        switch ($request->sort) {
            case 'newest':
                $query->orderBy('created_at', 'desc');
                break;

            case 'popular':
                $query->orderBy('total_students', 'desc');
                break;

            case 'price_low':
                // ✅ Free courses first, then paid ascending
                $query->orderByRaw("CASE WHEN price_type = 'free' THEN 0 ELSE 1 END")
                      ->orderBy('price', 'asc');
                break;

            case 'price_high':
                // ✅ Paid first, descending price
                $query->orderByRaw("CASE WHEN price_type = 'free' THEN 1 ELSE 0 END")
                      ->orderBy('price', 'desc');
                break;

            case 'duration':
                // ✅ Normalize duration types for sorting
                $query->orderByRaw("
                    CASE duration_type
                        WHEN 'hours' THEN duration_value
                        WHEN 'days' THEN duration_value * 24
                        WHEN 'weeks' THEN duration_value * 24 * 7
                        WHEN 'months' THEN duration_value * 24 * 30
                        ELSE duration_value
                    END ASC
                ");
                break;

            default:
                $query->orderBy('created_at', 'desc');
                break;
        }
    } else {
        $query->orderBy('created_at', 'desc'); // Default sort
    }

    // 📄 Pagination
    $courses = $query->paginate(12);

    $categories = CourseCategory::where('is_active', true)->get();

    return view('courses-quran', compact('courses', 'categories'));
}


public function coursesByCategory($slug)
{
    // Get category by slug
    $category = CourseCategory::where('slug', $slug)->where('is_active', true)->firstOrFail();

    // Get only active courses under this category
    $courses = Course::with('teacher', 'category')
        ->where('category_id', $category->id)
        ->where('is_active', true)
        ->orderBy('created_at', 'desc')
        ->paginate(12);

    // For sidebar or filters
    $categories = CourseCategory::where('is_active', true)->get();

    return view('courses-quran', compact('courses', 'categories', 'category'));
}





public function shows($slug)
{
    $course = Course::where('slug', $slug)->firstOrFail();
    return view('nadeem', compact('course'));
}

}
