@extends('admin.layouts.app')

@section('title', 'Dashboard')
@section('page-title', 'Dashboard')

<style>
/* Modern Glassmorphism Design */
.glassmorphism-card {
    background: rgba(255, 255, 255, 0.1);
    backdrop-filter: blur(20px);
    -webkit-backdrop-filter: blur(20px);
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 20px;
    padding: 25px;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.glassmorphism-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.15);
}

/* Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.stat-card {
    display: flex;
    align-items: center;
    padding: 25px;
    color: white;
    text-decoration: none;
    border-radius: 15px;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(135deg, rgba(255,255,255,0.1) 0%, rgba(255,255,255,0.05) 100%);
    z-index: 1;
}

.stat-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 15px 30px rgba(0, 0, 0, 0.2);
}

.stat-content {
    position: relative;
    z-index: 2;
    flex: 1;
}

.stat-icon {
    font-size: 2.5rem;
    margin-right: 20px;
    position: relative;
    z-index: 2;
}

.stat-number {
    font-size: 2.2rem;
    font-weight: 700;
    line-height: 1;
    margin-bottom: 5px;
}

.stat-label {
    font-size: 0.9rem;
    opacity: 0.9;
    font-weight: 500;
}

.stat-trend {
    font-size: 0.8rem;
    margin-top: 5px;
    display: flex;
    align-items: center;
    gap: 5px;
}

/* Color Themes */
.bg-primary { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
.bg-success { background: linear-gradient(135deg, #56ab2f 0%, #a8e6cf 100%); }
.bg-info { background: linear-gradient(135deg, #3498db 0%, #85c1e9 100%); }
.bg-warning { background: linear-gradient(135deg, #f39c12 0%, #f7dc6f 100%); }
.bg-danger { background: linear-gradient(135deg, #e74c3c 0%, #f1948a 100%); }
.bg-dark { background: linear-gradient(135deg, #2c3e50 0%, #566573 100%); }
.bg-purple { background: linear-gradient(135deg, #9b59b6 0%, #e8daef 100%); }
.bg-teal { background: linear-gradient(135deg, #1abc9c 0%, #a3e4d7 100%); }

/* Charts Section */
.charts-grid {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 20px;
    margin-bottom: 30px;
}

.chart-card {
    background: white;
    border-radius: 15px;
    padding: 25px;
    box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
}

/* Recent Activity */
.activity-list {
    list-style: none;
    padding: 0;
    margin: 0;
}

.activity-item {
    display: flex;
    align-items: center;
    padding: 15px 0;
    border-bottom: 1px solid rgba(0, 0, 0, 0.1);
}

.activity-item:last-child {
    border-bottom: none;
}

.activity-icon {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-right: 15px;
    font-size: 1rem;
}

.activity-content {
    flex: 1;
}

.activity-title {
    font-weight: 600;
    margin-bottom: 2px;
}

.activity-time {
    font-size: 0.8rem;
    color: #6c757d;
}

/* Quick Actions */
.actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
}

.action-btn {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 20px;
    border-radius: 12px;
    color: white;
    text-decoration: none;
    transition: all 0.3s ease;
    text-align: center;
}

.action-btn:hover {
    transform: translateY(-3px);
    color: white;
    text-decoration: none;
    box-shadow: 0 10px 20px rgba(0, 0, 0, 0.2);
}

.action-btn i {
    font-size: 1.8rem;
    margin-bottom: 10px;
}

.action-btn span {
    font-weight: 600;
    font-size: 0.9rem;
}

/* Responsive */
@media (max-width: 768px) {
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .charts-grid {
        grid-template-columns: 1fr;
    }
    
    .stat-number {
        font-size: 1.8rem;
    }
}
</style>

@section('content')

<!-- Hero Section -->
<div class="row mb-4">
    <div class="col-12">
        <div class="glassmorphism-card bg-success">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h1 class="text-white mb-2">Welcome to Quran Ezy Academy</h1>
                    <p class="text-white-50 mb-0">Manage your academy efficiently with real-time insights and analytics</p>
                </div>
                <div class="col-md-4 text-end">
                    <i class="fas fa-graduation-cap fa-4x text-white opacity-50"></i>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Main Stats -->
<div class="stats-grid">
    <a href="{{ route('admin.courses.index') }}" class="stat-card bg-primary">
        <div class="stat-icon">
            <i class="fas fa-book-open"></i>
        </div>
        <div class="stat-content">
            <div class="stat-number" data-count="{{ $stats['title'] }}">0</div>
            <div class="stat-label">Total Courses </div>
            <div class="stat-trend">
                <i class="fas fa-arrow-up text-success"></i>
                <span>{{ $courseStats['active_courses'] }} Active</span>
            </div>
        </div>
    </a>

    <a href="{{ route('admin.teachers.index') }}" class="stat-card bg-success">
        <div class="stat-icon">
            <i class="fas fa-chalkboard-teacher"></i>
        </div>
        <div class="stat-content">
            <div class="stat-number" data-count="{{ $stats['total_teachers'] }}">0</div>
            <div class="stat-label">Qualified Teachers</div>
        </div>
    </a>


    <a href="{{ route('admin.enrollments.index') }}" class="stat-card bg-warning">
        <div class="stat-icon">
            <i class="fas fa-clipboard-list"></i>
        </div>
        <div class="stat-content">
            <div class="stat-number" data-count="{{ $stats['total_enrollments'] }}">0</div>
            <div class="stat-label">Total Enrollments</div>
            <div class="stat-trend">
                <span class="text-warning">{{ $stats['total_pending_enrollments'] }} Pending</span>
            </div>
        </div>
    </a>
</div>

<!-- Second Row Stats -->
<div class="stats-grid mb-4">
    <a href="{{ route('admin.admin-users.index') }}" class="stat-card bg-dark">
        <div class="stat-icon">
            <i class="fas fa-user-shield"></i>
        </div>
        <div class="stat-content">
            <div class="stat-number" data-count="{{ $stats['total_admins'] }}">0</div>
            <div class="stat-label">Admin Users</div>
        </div>
    </a>



    <a href="{{ route('admin.faqs.index') }}" class="stat-card bg-teal">
        <div class="stat-icon">
            <i class="fas fa-question-circle"></i>
        </div>
        <div class="stat-content">
            <div class="stat-number" data-count="{{ $stats['total_faqs'] }}">0</div>
            <div class="stat-label">FAQs</div>
        </div>
    </a>

    <div class="stat-card bg-danger">
        <div class="stat-icon">
            <i class="fas fa-dollar-sign"></i>
        </div>
        <div class="stat-content">
            <div class="stat-number">$<span data-count="{{ $stats['revenue'] }}">0</span></div>
            <div class="stat-label">Total Revenue</div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Recent Enrollments -->
    <div class="col-lg-8 mb-4">
        <div class="chart-card">
            <h5 class="card-title mb-4">
                <i class="fas fa-clock text-primary me-2"></i>
                Recent Enrollments
            </h5>
            <div class="activity-list">
                @forelse($recentEnrollments as $enrollment)
                <div class="activity-item">
                    <div class="activity-icon bg-primary text-white">
                        <i class="fas fa-user-plus"></i>
                    </div>
                    <div class="activity-content">
                        <div class="activity-title">
                            {{ $enrollment->first_name }} {{ $enrollment->last_name }}
                        </div>
                        <div class="activity-desc text-muted">
                            Enrolled in {{ $enrollment->course->title ?? 'N/A' }}
                        </div>
                        <div class="activity-time">
                            {{ $enrollment->created_at->diffForHumans() }}
                            <span class="badge bg-{{ $enrollment->status == 'pending' ? 'warning' : 'success' }} ms-2">
                                {{ ucfirst($enrollment->status) }}
                            </span>
                        </div>
                    </div>
                </div>
                @empty
                <div class="text-center py-4 text-muted">
                    <i class="fas fa-inbox fa-2x mb-2"></i>
                    <p>No recent enrollments</p>
                </div>
                @endforelse
            </div>
        </div>
    </div>

    <!-- Quick Actions -->
    <div class="col-lg-4 mb-4">
        <div class="chart-card">
            <h5 class="card-title mb-4">
                <i class="fas fa-bolt text-warning me-2"></i>
                Quick Actions
            </h5>
            <div class="actions-grid">
                <a href="{{ route('admin.courses.create') }}" class="action-btn bg-primary">
                    <i class="fas fa-plus-circle"></i>
                    <span>Add Course</span>
                </a>
                <a href="{{ route('admin.teachers.create') }}" class="action-btn bg-success">
                    <i class="fas fa-user-plus"></i>
                    <span>Add Teacher</span>
                </a>
                <a href="{{ route('admin.enrollments.index') }}" class="action-btn bg-info">
                    <i class="fas fa-clipboard-check"></i>
                    <span>New Enrollment</span>
                </a>
                <a href="{{ route('admin.faqs.create') }}" class="action-btn bg-warning">
                    <i class="fas fa-question"></i>
                    <span>Add FAQ</span>
                </a>
            </div>
        </div>
    </div>
</div>

@endsection

<script>
// Animated counters
function animateCounters() {
    const counters = document.querySelectorAll('.stat-number');
    
    counters.forEach(counter => {
        const target = parseInt(counter.dataset.count) || parseFloat(counter.dataset.count) || 0;
        const increment = target / 50;
        let current = 0;
        
        const updateCounter = () => {
            if (current < target) {
                current += increment;
                if (counter.innerHTML.includes('$')) {
                    counter.innerHTML = '$' + Math.ceil(current).toLocaleString();
                } else {
                    counter.textContent = Math.ceil(current).toLocaleString();
                }
                setTimeout(updateCounter, 30);
            } else {
                if (counter.innerHTML.includes('$')) {
                    counter.innerHTML = '$' + target.toLocaleString();
                } else {
                    counter.textContent = target.toLocaleString();
                }
            }
        };
        
        updateCounter();
    });
}

// Initialize when page loads
document.addEventListener('DOMContentLoaded', function() {
    animateCounters();
});
</script>
