@extends('admin.layouts.app')

@section('title', 'Edit Specialty')
@section('page-title', 'Edit Specialty')

@section('content')
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="fas fa-edit me-2"></i>
                    Edit Specialty
                </h5>
            </div>
            <div class="card-body">
                <form action="{{ route('admin.specialties.update', $specialty) }}" method="POST" enctype="multipart/form-data">
                    @csrf
                    @method('PUT')
                    
                    <div class="mb-3">
                        <label for="title" class="form-label">Title <span class="text-danger">*</span></label>
                        <input type="text" 
                               class="form-control @error('title') is-invalid @enderror" 
                               id="title" 
                               name="title" 
                               value="{{ old('title', $specialty->title) }}" 
                               placeholder="Enter specialty title"
                               required>
                        @error('title')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <div class="mb-3">
                        <label for="description" class="form-label">Description <span class="text-danger">*</span></label>
                        <textarea class="form-control @error('description') is-invalid @enderror" 
                                  id="description" 
                                  name="description" 
                                  rows="6" 
                                  placeholder="Enter detailed description of this specialty"
                                  required>{{ old('description', $specialty->description) }}</textarea>
                        @error('description')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <div class="form-text">Describe what makes this specialty unique and the services provided.</div>
                    </div>

                    <div class="mb-3">
                        <label for="image" class="form-label">Specialty Image</label>
                        @if($specialty->image)
                            <div class="mb-2">
                                <img src="{{ Storage::url($specialty->image) }}" 
                                     alt="{{ $specialty->title }}" 
                                     class="img-thumbnail" 
                                     style="max-width: 200px; max-height: 150px;">
                                <div class="form-text">Current image</div>
                            </div>
                        @endif
                        <input type="file" 
                               class="form-control @error('image') is-invalid @enderror" 
                               id="image" 
                               name="image" 
                               accept="image/*">
                        @error('image')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <div class="form-text">
                            Supported formats: JPG, PNG, GIF. Max size: 2MB. Recommended size: 400x300px
                            @if($specialty->image)
                                <br>Leave empty to keep current image
                            @endif
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="is_active" class="form-label">Status</label>
                        <select class="form-select @error('is_active') is-invalid @enderror" 
                                id="is_active" 
                                name="is_active">
                            <option value="1" {{ old('is_active', $specialty->is_active) == '1' ? 'selected' : '' }}>Active</option>
                            <option value="0" {{ old('is_active', $specialty->is_active) == '0' ? 'selected' : '' }}>Inactive</option>
                        </select>
                        @error('is_active')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <div class="form-text">Only active specialties will be displayed on the website.</div>
                    </div>

                    <div class="card bg-light mb-3">
                        <div class="card-header">
                            <h6 class="mb-0">
                                <i class="fas fa-eye me-2"></i>Preview
                            </h6>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-4">
                                    @if($specialty->image)
                                        <img src="{{ Storage::url($specialty->image) }}" 
                                             alt="{{ $specialty->title }}" 
                                             class="img-fluid rounded"
                                             id="preview-image">
                                    @else
                                        <div class="bg-secondary rounded d-flex align-items-center justify-content-center" style="height: 200px;" id="preview-image-placeholder">
                                            <i class="fas fa-image fa-3x text-white"></i>
                                        </div>
                                    @endif
                                </div>
                                <div class="col-md-8">
                                    <h5 id="preview-title">{{ $specialty->title }}</h5>
                                    <p class="text-muted" id="preview-description">{{ Str::limit($specialty->description, 150) }}</p>
                                    <button class="btn btn-primary btn-sm" disabled>Learn More</button>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="d-flex justify-content-between">
                        <a href="{{ route('admin.specialties.index') }}" class="btn btn-secondary">
                            <i class="fas fa-arrow-left me-2"></i>Back to List
                        </a>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Update Specialty
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
// Live preview update
document.getElementById('title').addEventListener('input', function() {
    document.getElementById('preview-title').textContent = this.value || 'Specialty Title';
});

document.getElementById('description').addEventListener('input', function() {
    const text = this.value || 'Specialty description will appear here...';
    document.getElementById('preview-description').textContent = text.length > 150 ? text.substring(0, 150) + '...' : text;
});

// Image preview
document.getElementById('image').addEventListener('change', function(e) {
    const file = e.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            const previewImage = document.getElementById('preview-image');
            const placeholder = document.getElementById('preview-image-placeholder');
            
            if (previewImage) {
                previewImage.src = e.target.result;
            } else if (placeholder) {
                placeholder.outerHTML = '<img src="' + e.target.result + '" alt="Preview" class="img-fluid rounded" id="preview-image">';
            }
        };
        reader.readAsDataURL(file);
    }
});
</script>
@endsection

